# -*- coding: utf-8 -*-
"""
Mapping Module Methods ArcGIS Pro

Nathan Jennings
Created on:  03.02.2019
Updated on:  02.21.2021

02.24.2021

Updated comments for getLayerExtent and setExtent - see inline comments

# getLayerExtent
# getLayerExtent is correct with "zoom to selected features, etc"
# See https://pro.arcgis.com/en/pro-app/latest/arcpy/mapping/mapframe-class.htm
# for details on getLayerExtent(layer, {selection_only}, {symbolized_extent})
# {selection_only} parameter defaults to TRUE

# setExtent
# setExtent is corret
# See https://pro.arcgis.com/en/pro-app/latest/arcpy/mapping/camera-class.htm
# for setExtent(extent)


"""

print ("Demo 9b Pro - Mapping Module Methods\n")

import arcpy, sys, traceback, datetime
# NOTE:  - no special mapping module needs importing
#from arcpy.mapping import *   # Eliminates the need to write arcpy.mapping
                              # before each mapping module related class, function
                              # or method

author = "N. Jennings"

# reformat date to MM.DD.YYYY
# NOTE: using a lowercase %y will result in MM.DD.YY format
# See Python.org or text regarding the datetime module

CUR_DATE = datetime.date.today().strftime('%m.%d.%Y')

try:

    # Get a pre-existing  Pro project file
    # This project file also contains the pre-configured layers, Map tabs, Layout tabs, and map elements

    #ppath = "C:\\Users\\Nate\\Documents\\ArcGIS\\Projects\\ProMapModule\\"
    ppath = "C:\\Users\\Nate\\Python\\Chapter9_DemoScriptsandData\\Chapter09\\ProMapModule\\ProMapModule\\"
    
    
    # output for PDFs
    # Also change as required.  The folder does not have to be c:\temp  
    mappath = "c:\\temp\\"

    # The full path to the Pro project file
    aprx = arcpy.mp.ArcGISProject(ppath + "ProMapModule.aprx")
    
    # get a list of Maps that contain layers
    # [0] inicates the first (or only map from the map list, in this case)
    
    maplist = aprx.listMaps("Map") [0]  # the name of the Map tab is "Map"
    layout = aprx.listLayouts("Layout")[0] # the name of the Layout tab is "Layout"
    mapframe = layout.listElements("MAPFRAME_ELEMENT", "Map Frame")[0]  # the name of the map frame name is "Map Frame"  
    tElements = layout.listElements("TEXT_ELEMENT") # get a list of only "text" type element
    
    # get a list of layers
    for TOCLayer in maplist.listLayers():
       
        
        if TOCLayer.longName == "Detail Map\\Neighborhoods":
            query = """"NAME" = 'Alkali Flat'"""
                        
        
            arcpy.SelectLayerByAttribute_management(TOCLayer, "NEW_SELECTION", query)
            
            result = arcpy.GetCount_management(TOCLayer)
            print ("Number of selected features: " + str(result))

            # selection defaults to TRUE
            # getLayerExtent is correct with "zoom to selected features, etc"
            # See https://pro.arcgis.com/en/pro-app/latest/arcpy/mapping/mapframe-class.htm
            # for details on getLayerExtent(layer, {selection_only}, {symbolized_extent})
            # setExtent is corret
            # See https://pro.arcgis.com/en/pro-app/latest/arcpy/mapping/camera-class.htm
            # for setExtent(extent)
            layer_extent = mapframe.getLayerExtent(TOCLayer) 
                                    
            mapframe.camera.setExtent(layer_extent)
    
            for tElement in tElements:
                
                if tElement.name == 'Map Title':
                    tElement.text = 'Alkali Flat'
                
                if tElement.name == 'Print Date':
                    tElement.text = str(CUR_DATE)

            print ('Exporting Map 1...')
            if arcpy.Exists(mappath + 'selected_feature_Map1.pdf'):
                arcpy.Delete_management(mappath + 'selected_feature_Map1.pdf')     
            layout.exportToPDF(mappath + "selected_feature_Map1.pdf")
            
            # Map 2
            # clear any selected features        
            arcpy.SelectLayerByAttribute_management(TOCLayer, "CLEAR_SELECTION")

            print ('Exporting Map 2...')
            if arcpy.Exists(mappath + 'no_selected_features_Map2.pdf'):
                arcpy.Delete_management(mappath + 'no_selected_features_Map2.pdf')     
            layout.exportToPDF(mappath + "no_selected_features_Map2.pdf")
            
            # Map 3
            query = """"NAME" = 'Downtown'"""
            print (query)
        
            arcpy.SelectLayerByAttribute_management(TOCLayer, "NEW_SELECTION", query)
            layer_extent = mapframe.getLayerExtent(TOCLayer)
            mapframe.camera.setExtent(layer_extent)
            
            for tElement in tElements:
    
                # if the text element name is 'Map Title',
                # then assign a specific name for the title
                # see the elements properties in ArcMap (under size and position)
                
                if tElement.name == 'Map Title':
                    
                    tElement.text = 'Downtown'
    
            print ('Exporting Map 3...')
            if arcpy.Exists(mappath + 'GetSelectedExtentfromLayer_Map3.pdf'):
                arcpy.Delete_management(mappath + 'GetSelectedExtentfromLayer_Map3.pdf')
            
            layout.exportToPDF(mappath + 'GetSelectedExtentfromLayer_Map3.pdf')
    
            # Clear feature selection to remove the "highlighted features"
    
            # Map 4
            # clear any selected features, but don't change the zoom extent        
            arcpy.SelectLayerByAttribute_management(TOCLayer, "CLEAR_SELECTION")
    
            print ('Exporting Map 4...')
            if arcpy.Exists(mappath + 'GetSelectedExtentfromLayer_NoHighlight_Map4.pdf'):
                arcpy.Delete_management(mappath + 'GetSelectedExtentfromLayer_NoHighlight_Map4.pdf')
            
            layout.exportToPDF(mappath + 'GetSelectedExtentfromLayer_NoHighlight_Map4.pdf')
    
            # Map 5
    
            TOCLayer.definitionQuery = query  # Assign the existing query to the
                                              # definitionQuery property of the layer
            
            #layer_extent = mapframe.getLayerExtent(TOCLayer)
            #mapframe.camera.setExtent(layer_extent)
            mapframe.camera.scale = mapframe.camera.scale * 1.1
            
            print ('Exporting Map 5...')
            if arcpy.Exists(mappath + 'DefinitionQuery_ScaleChange_Map5.pdf'):
                arcpy.Delete_management(mappath + 'DefinitionQuery_ScaleChange_Map5.pdf')
            
            layout.exportToPDF(mappath + 'DefinitionQuery_ScaleChange_Map5.pdf')
    
            # Map 6
            query = """"NAME" = 'Newton Booth'"""
    
            TOCLayer.definitionQuery = query   # Assign the existing query to the
                                               # definitionQuery property of the layer
    
            # get the extent of the features (in this case the definition queried features
            # assign this extent to the data frame extent
            layer_extent = mapframe.getLayerExtent(TOCLayer)
            mapframe.camera.setExtent(layer_extent)
            mapframe.camera.scale = mapframe.camera.scale * 1.1   # Add a 10% buffer to the extent
    
            for tElement in tElements:
    
                # if the text element name is 'Map Title',
                # then assign a specific name for the title
                # see the elements properties in Pro (right click on properties, see General section)
                
                if tElement.name == 'Map Title':
                    
                    tElement.text = 'Newton Booth'
    
            # The exported map will contain the definition query
            # the modified data frame scale
            print ('Exporting Map 6...')
            if arcpy.Exists(mappath + 'DefinitionQuery_GetExtent_ScaleChange_Map6.pdf'):
                arcpy.Delete_management(mappath + 'DefinitionQuery_GetExtent_ScaleChange_Map6.pdf')
            
            layout.exportToPDF(mappath + 'DefinitionQuery_GetExtent_ScaleChange_Map6.pdf')
            
            # Map 7
            # Add a new layer to the map
            
            legend = layout.listElements("LEGEND_ELEMENT", "Legend")[0]
            
            cityFacilitiesLayer = arcpy.mp.LayerFile(ppath + "City_Facilities.lyrx")
            
            
            
            # Get the city facilities.lyr file from disk
            # The layer "auto adds" to the legend
            # Add the layer to the map frame
            # Turn labels on for the city facilities layer based on the predefined
            # label attribute in the .lyrx file
            # Save the new layer changes (labels turned on) to a new layer (.lyrx) file
            # Export a new map

            legend = layout.listElements("LEGEND_ELEMENT") [0] # Get the first and only legend

            maplist.addLayer(cityFacilitiesLayer)
            
            NewLayerList = maplist.listLayers()
            
            for lyr in NewLayerList:
                
                if lyr.name == 'City_Facilities':
                    
                    lyr.showLabels = True

                    if arcpy.Exists(ppath + "City_Facilities_Labels.lyrx"):
                        arcpy.Delete_management(ppath + "City_Facilities_Labels.lyrx")
                        
                    lyr.saveACopy(ppath + "City_Facilities_Labels.lyrx")
            
            # Save the layer file to a new .lyrx file name
            # The layer saveACopy only works for any layer that is a .lyrx file
            # save and saveACopy cannot be used with other layers in the Table of Contents
            # that do not originate from a .lyrc file
            

            print ('Exporting Map 7...')
            if arcpy.Exists(mappath + 'City_Facilities_Map7.pdf'):
                arcpy.Delete_management(mappath + 'City_Facilities_Map7.pdf')
            
            layout.exportToPDF(mappath + 'City_Facilities_Map7.pdf')

            
                
    print ('Completed Map Updates')


except:
     # https://pro.arcgis.com/en/pro-app/arcpy/get-started/error-handling-with-python.htm
    
    tb = sys.exc_info()[2]
    tbinfo = traceback.format_tb(tb)[0]
    pymsg = "PYTHON ERRORS:\nTraceback Info:\n" + tbinfo + "\nError Info:\n" + str(sys.exc_info()[1])
    msgs = "ARCPY ERRORS:\n" + arcpy.GetMessages(2) + "\n"

    arcpy.AddError(msgs)
    arcpy.AddError(pymsg)

    print (msgs)
    print (pymsg)
    
    arcpy.AddMessage(arcpy.GetMessages(1))
    print (arcpy.GetMessages(1))
